<?php
// helpers/code128.php
// Code128 (Subset B) -> SVG renderer

function code128_patterns(): array {
    // Each entry is 6 digits (bar/space alternating) total 11 modules. Stop is 7 digits.
    return [
        "212222","222122","222221","121223","121322","131222","122213","122312","132212","221213",
        "221312","231212","112232","122132","122231","113222","123122","123221","223211","221132",
        "221231","213212","223112","312131","311222","321122","321221","312212","322112","322211",
        "212123","212321","232121","111323","131123","131321","112313","132113","132311","211313",
        "231113","231311","112133","112331","132131","113123","113321","133121","313121","211331",
        "231131","213113","213311","213131","311123","311321","331121","312113","312311","332111",
        "314111","221411","431111","111224","111422","121124","121421","141122","141221","112214",
        "112412","122114","122411","142112","142211","241211","221114","413111","241112","134111",
        "111242","121142","121241","114212","124112","124211","411212","421112","421211","212141",
        "214121","412121","111143","111341","131141","114113","114311","411113","411311","113141",
        "114131","311141","411131","211412","211214","211232","2331112"
    ];
}

function code128_encode_subset_b(string $text): array {
    $text = (string)$text;

    for ($i = 0; $i < strlen($text); $i++) {
        $o = ord($text[$i]);
        if ($o < 32 || $o > 126) {
            throw new Exception("Unsupported character for Code128-B. Use ASCII 32..126 only.");
        }
    }

    $patterns = code128_patterns();

    $start_code = 104; // Start B
    $codes = [$start_code];

    for ($i = 0; $i < strlen($text); $i++) {
        $codes[] = ord($text[$i]) - 32;
    }

    $checksum = $start_code;
    for ($i = 1; $i < count($codes); $i++) {
        $checksum += $codes[$i] * $i;
    }
    $checksum = $checksum % 103;

    $codes[] = $checksum;
    $codes[] = 106; // stop

    $seq = [];
    foreach ($codes as $c) {
        $pat = $patterns[$c] ?? null;
        if ($pat === null) throw new Exception("Invalid Code128 pattern index: $c");
        $seq[] = $pat;
    }

    return $seq;
}

/**
 * Generate SVG for Code128-B.
 *
 * IMPORTANT for camera scanning:
 * - Use module=3 (thicker bars)
 * - Use height=80+ (taller)
 */
function code128_svg(string $text, int $height = 90, int $module = 3, int $quiet = 20): string {
    $seq = code128_encode_subset_b($text);

    $totalModules = 0;
    foreach ($seq as $pat) {
        for ($i = 0; $i < strlen($pat); $i++) {
            $totalModules += (int)$pat[$i];
        }
    }

    $widthPx = ($totalModules * $module) + ($quiet * 2);

    $x = $quiet;
    $bars = [];

    foreach ($seq as $pat) {
        $isBar = true;
        for ($i = 0; $i < strlen($pat); $i++) {
            $w = (int)$pat[$i] * $module;
            if ($isBar) {
                $bars[] = '<rect x="' . (int)$x . '" y="0" width="' . (int)$w . '" height="' . (int)$height . '" />';
            }
            $x += $w;
            $isBar = !$isBar;
        }
    }

    return
        '<svg xmlns="http://www.w3.org/2000/svg" width="' . $widthPx . '" height="' . $height . '" viewBox="0 0 ' . $widthPx . ' ' . $height . '">' .
        '<rect x="0" y="0" width="' . $widthPx . '" height="' . $height . '" fill="#fff"/>' .
        '<g fill="#000">' . implode('', $bars) . '</g>' .
        '</svg>';
}
