<?php
// public/print_label.php
require_once __DIR__ . '/../helpers/auth.php';
require_role(['admin', 'staff']);

require_once __DIR__ . '/../helpers/code128.php';

$pdo = db();

$id  = (int)($_GET['id'] ?? 0);
$sku = trim((string)($_GET['sku'] ?? ''));

// mode: qr | barcode | both
$mode = strtolower(trim((string)($_GET['mode'] ?? 'both')));
if (!in_array($mode, ['qr', 'barcode', 'both'], true)) {
    $mode = 'both';
}
$showBarcode = ($mode === 'barcode' || $mode === 'both');
$showQr      = ($mode === 'qr' || $mode === 'both');

$product = null;

if ($id > 0) {
    $stmt = $pdo->prepare("SELECT id, sku, name, barcode_value FROM products WHERE id = :id LIMIT 1");
    $stmt->execute([':id' => $id]);
    $product = $stmt->fetch();
} elseif ($sku !== '') {
    $stmt = $pdo->prepare("SELECT id, sku, name, barcode_value FROM products WHERE sku = :sku LIMIT 1");
    $stmt->execute([':sku' => $sku]);
    $product = $stmt->fetch();
}

// Prefer packaging barcode_value; fallback to internal SKU
$barcodeText = '';
if ($product) {
    $barcodeText = trim((string)($product['barcode_value'] ?? ''));
    if ($barcodeText === '') $barcodeText = (string)$product['sku'];
}

// Code128 barcode SVG
$barcodeSvg = '';
if ($product && $showBarcode) {
    try {
        $barcodeSvg = code128_svg($barcodeText, 70, 3, 18);
    } catch (Throwable $e) {
        $barcodeSvg = '';
    }
}

?><!doctype html>
<html>
<head>
  <meta charset="utf-8">
  <title>Print Label</title>
  <style>
    body { font-family: Arial, sans-serif; margin: 18px; }
    .label {
      width: 80mm;
      border: 1px dashed #bbb;
      padding: 10px;
      margin: 0 auto;
      text-align: center;
      background: #fff;
    }
    .name { font-size: 14px; font-weight: 700; margin-bottom: 8px; }
    .section-title {
      font-size: 11px;
      font-weight: 700;
      color: #444;
      margin-top: 10px;
      margin-bottom: 6px;
      text-transform: uppercase;
      letter-spacing: 0.5px;
    }
    .barcode-wrap svg {
      width: 74mm;
      height: 22mm;
      display: block;
      margin: 0 auto;
    }
    .barcode-text {
      font-size: 12px;
      font-weight: 700;
      margin-top: 6px;
      letter-spacing: 1px;
    }
    .qr-img {
      width: 42mm;
      height: 42mm;
      object-fit: contain;
      display: block;
      margin: 0 auto;
    }
    .sku {
      font-size: 12px;
      margin-top: 8px;
      font-weight: 700;
      letter-spacing: 1px;
    }
    .tip { max-width: 80mm; margin: 12px auto 0; color: #666; font-size: 12px; }
    .actions { text-align: center; margin-top: 12px; }
    button { padding: 8px 12px; cursor: pointer; }
    @media print {
      body { margin: 0; }
      .actions, .tip { display: none; }
      .label { border: none; margin: 0; }
    }
  </style>
</head>
<body>

<?php if (!$product): ?>
  <h3>Print Label</h3>
  <p>No product selected.</p>
  <p>Use:</p>
  <ul>
    <li><b>print_label.php?id=123</b></li>
    <li><b>print_label.php?sku=PRD-000123</b></li>
    <li><b>print_label.php?id=123&amp;mode=barcode</b> (barcode only)</li>
    <li><b>print_label.php?id=123&amp;mode=qr</b> (QR only)</li>
    <li><b>print_label.php?id=123&amp;mode=both</b> (default)</li>
  </ul>
<?php else: ?>
  <?php
    $qr_payload = (string)$product['sku'];
    $qr_url = "https://api.qrserver.com/v1/create-qr-code/?size=280x280&data=" . urlencode($qr_payload);
  ?>

  <div class="label">
    <div class="name"><?= htmlspecialchars((string)$product['name']) ?></div>

    <?php if ($showBarcode): ?>
      <div class="section-title">Barcode (scan this with USB barcode scanner)</div>
      <?php if ($barcodeSvg !== ''): ?>
        <div class="barcode-wrap"><?= $barcodeSvg ?></div>
        <div class="barcode-text"><?= htmlspecialchars($barcodeText) ?></div>
      <?php else: ?>
        <div class="text-danger small">Failed to render barcode.</div>
      <?php endif; ?>
    <?php endif; ?>

    <?php if ($showQr): ?>
      <div class="section-title">QR (phone camera)</div>
      <div>
        <img class="qr-img" src="<?= htmlspecialchars($qr_url) ?>" alt="QR Code">
      </div>
    <?php endif; ?>

    <div class="sku">SKU: <?= htmlspecialchars((string)$product['sku']) ?></div>
  </div>

  <div class="actions">
    <button onclick="window.print()">Print</button>
  </div>

  <div class="tip">
    For best scanning: print on matte paper, avoid glare, and keep codes large and sharp.
  </div>
<?php endif; ?>

</body>
</html>
