<?php
// api/delivery_status.php
require_once __DIR__ . '/../helpers/auth.php';
require_role(['driver']);

header('Content-Type: application/json');

$pdo = db();
$user = current_user();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['ok' => false, 'error' => 'Method not allowed']);
    exit;
}

$raw = file_get_contents('php://input');
$data = json_decode($raw, true);
if (!is_array($data)) {
    http_response_code(400);
    echo json_encode(['ok' => false, 'error' => 'Invalid JSON']);
    exit;
}

$delivery_id = (int)($data['delivery_id'] ?? 0);
$new_status  = trim((string)($data['status'] ?? ''));

// Driver allowed transitions
$allowed_driver_statuses = ['PICKED_UP', 'IN_TRANSIT', 'DELIVERED'];

if ($delivery_id <= 0 || !in_array($new_status, $allowed_driver_statuses, true)) {
    http_response_code(400);
    echo json_encode(['ok' => false, 'error' => 'Invalid delivery_id/status']);
    exit;
}

// Validate delivery & assignment
$stmt = $pdo->prepare("
  SELECT id, assigned_driver_id, status
  FROM deliveries
  WHERE id = :id
  LIMIT 1
");
$stmt->execute([':id' => $delivery_id]);
$delivery = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$delivery) {
    http_response_code(404);
    echo json_encode(['ok' => false, 'error' => 'Delivery not found']);
    exit;
}

if ((int)$delivery['assigned_driver_id'] !== (int)$user['id']) {
    http_response_code(403);
    echo json_encode(['ok' => false, 'error' => 'Not assigned to this delivery']);
    exit;
}

if (in_array($delivery['status'], ['DELIVERED','CANCELLED'], true)) {
    http_response_code(400);
    echo json_encode(['ok' => false, 'error' => 'Delivery already closed']);
    exit;
}

// Enforce simple status progression
$progress = ['PENDING' => 0, 'ASSIGNED' => 1, 'PICKED_UP' => 2, 'IN_TRANSIT' => 3, 'DELIVERED' => 4];
$currentLevel = $progress[$delivery['status']] ?? 0;
$newLevel     = $progress[$new_status] ?? 0;

if ($newLevel <= $currentLevel) {
    http_response_code(400);
    echo json_encode(['ok' => false, 'error' => 'Cannot move status backward or same']);
    exit;
}

// ✅ POD REQUIRED before DELIVERED
if ($new_status === 'DELIVERED') {
    $pod = $pdo->prepare("
      SELECT id
      FROM delivery_proofs
      WHERE delivery_id = :did AND driver_id = :uid
      ORDER BY created_at DESC
      LIMIT 1
    ");
    $pod->execute([':did' => $delivery_id, ':uid' => (int)$user['id']]);

    if (!$pod->fetch(PDO::FETCH_ASSOC)) {
        http_response_code(400);
        echo json_encode([
            'ok' => false,
            'error' => 'Proof of Delivery (POD) is required before marking DELIVERED.'
        ]);
        exit;
    }
}

try {
    $extraSql = "";
    if ($new_status === 'PICKED_UP') $extraSql = ", picked_up_at = NOW()";
    if ($new_status === 'DELIVERED') $extraSql = ", delivered_at = NOW()";

    $upd = $pdo->prepare("UPDATE deliveries SET status = :st $extraSql WHERE id = :id LIMIT 1");
    $upd->execute([':st' => $new_status, ':id' => $delivery_id]);

    echo json_encode([
        'ok' => true,
        'status' => $new_status,
        'updated_at' => date('Y-m-d H:i:s')
    ]);
} catch (Throwable $t) {
    http_response_code(500);
    echo json_encode(['ok' => false, 'error' => $t->getMessage()]);
}
