<?php
// api/location_update.php
require_once __DIR__ . '/../helpers/auth.php';

// Only logged-in drivers should use this endpoint
require_role(['driver']);

header('Content-Type: application/json');

$pdo = db();
$user = current_user();

// Allow only POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['ok' => false, 'error' => 'Method not allowed']);
    exit;
}

// Read JSON body
$raw = file_get_contents('php://input');
$data = json_decode($raw, true);

if (!is_array($data)) {
    http_response_code(400);
    echo json_encode(['ok' => false, 'error' => 'Invalid JSON']);
    exit;
}

$delivery_id = (int)($data['delivery_id'] ?? 0);
$lat = $data['lat'] ?? null;
$lng = $data['lng'] ?? null;

$accuracy = $data['accuracy_m'] ?? null;
$speed = $data['speed_mps'] ?? null;
$heading = $data['heading_deg'] ?? null;

// Basic validation
if ($delivery_id <= 0 || $lat === null || $lng === null) {
    http_response_code(400);
    echo json_encode(['ok' => false, 'error' => 'Missing delivery_id/lat/lng']);
    exit;
}

// Check that this driver is assigned to that delivery
$stmt = $pdo->prepare("
  SELECT id, assigned_driver_id, status
  FROM deliveries
  WHERE id = :id
  LIMIT 1
");
$stmt->execute([':id' => $delivery_id]);
$delivery = $stmt->fetch();

if (!$delivery) {
    http_response_code(404);
    echo json_encode(['ok' => false, 'error' => 'Delivery not found']);
    exit;
}

if ((int)$delivery['assigned_driver_id'] !== (int)$user['id']) {
    http_response_code(403);
    echo json_encode(['ok' => false, 'error' => 'Not assigned to this delivery']);
    exit;
}

if (in_array($delivery['status'], ['DELIVERED','CANCELLED'], true)) {
    http_response_code(400);
    echo json_encode(['ok' => false, 'error' => 'Delivery already closed']);
    exit;
}

// Save location
try {
    $ins = $pdo->prepare("
      INSERT INTO delivery_locations
      (delivery_id, driver_id, lat, lng, accuracy_m, speed_mps, heading_deg)
      VALUES
      (:delivery_id, :driver_id, :lat, :lng, :accuracy_m, :speed_mps, :heading_deg)
    ");
    $ins->execute([
        ':delivery_id' => $delivery_id,
        ':driver_id' => (int)$user['id'],
        ':lat' => (float)$lat,
        ':lng' => (float)$lng,
        ':accuracy_m' => ($accuracy !== null ? (float)$accuracy : null),
        ':speed_mps' => ($speed !== null ? (float)$speed : null),
        ':heading_deg' => ($heading !== null ? (float)$heading : null),
    ]);

    echo json_encode(['ok' => true, 'saved_at' => date('Y-m-d H:i:s')]);
} catch (Throwable $t) {
    http_response_code(500);
    echo json_encode(['ok' => false, 'error' => $t->getMessage()]);
}
