<?php
// public/stock_out.php
require_once __DIR__ . '/../helpers/auth.php';
require_role(['admin', 'staff']);

$pdo = db();
$title = "Stock OUT - IMS";
$user = current_user();

function post_str(string $key): string {
    return trim((string)($_POST[$key] ?? ''));
}
function post_int(string $key): int {
    return (int)($_POST[$key] ?? 0);
}

$error = '';
$success = '';

$products = $pdo->query("SELECT id, sku, name FROM products WHERE is_active=1 ORDER BY name ASC")->fetchAll();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $product_id = post_int('product_id');
    $qty = post_int('quantity');
    $notes = post_str('notes');

    if ($product_id <= 0 || $qty <= 0) {
        $error = "Please select a product and enter a quantity greater than 0.";
    } else {
        // Optional safety: prevent negative stock (we'll compute current stock quickly)
        $stmt = $pdo->prepare("
            SELECT
              COALESCE(SUM(CASE WHEN movement_type='IN' THEN quantity ELSE 0 END),0) -
              COALESCE(SUM(CASE WHEN movement_type='OUT' THEN quantity ELSE 0 END),0) -
              COALESCE(SUM(CASE WHEN movement_type='ADJUST' THEN quantity ELSE 0 END),0) AS stock
            FROM inventory_movements
            WHERE product_id = :pid
        ");
        $stmt->execute([':pid' => $product_id]);
        $current_stock = (int)($stmt->fetch()['stock'] ?? 0);

        if ($qty > $current_stock) {
            $error = "Not enough stock. Current stock: {$current_stock}";
        } else {
            try {
                $stmt = $pdo->prepare("
                    INSERT INTO inventory_movements
                    (product_id, movement_type, quantity, unit_cost, reference_type, reference_id, notes, created_by)
                    VALUES
                    (:product_id, 'OUT', :quantity, NULL, 'MANUAL', NULL, :notes, :created_by)
                ");
                $stmt->execute([
                    ':product_id' => $product_id,
                    ':quantity' => $qty,
                    ':notes' => $notes !== '' ? $notes : null,
                    ':created_by' => $user['id'],
                ]);

                $success = "Stock deducted successfully.";
            } catch (PDOException $e) {
                $error = "Error: " . $e->getMessage();
            }
        }
    }
}

// Recent OUT movements
$recent = $pdo->query("
    SELECT m.*, p.sku, p.name
    FROM inventory_movements m
    JOIN products p ON p.id = m.product_id
    WHERE m.movement_type='OUT'
    ORDER BY m.created_at DESC
    LIMIT 20
")->fetchAll();

require_once __DIR__ . '/../templates/header.php';
require_once __DIR__ . '/../templates/sidebar.php';
?>

<main class="col-12 col-md-9 col-lg-10 p-3 p-md-4">
  <h4 class="mb-3">Stock OUT</h4>

  <?php if ($error !== ''): ?>
    <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
  <?php endif; ?>

  <?php if ($success !== ''): ?>
    <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
  <?php endif; ?>

  <div class="row g-3">
    <div class="col-12 col-lg-4">
      <div class="card shadow-sm">
        <div class="card-body">
          <h5 class="card-title mb-3">Deduct Stock</h5>

          <form method="POST">
            <div class="mb-2">
              <label class="form-label">Product</label>
              <select class="form-select" name="product_id" required>
                <option value="0">-- Select --</option>
                <?php foreach ($products as $p): ?>
                  <option value="<?= (int)$p['id'] ?>">
                    <?= htmlspecialchars($p['name']) ?> (<?= htmlspecialchars($p['sku']) ?>)
                  </option>
                <?php endforeach; ?>
              </select>
            </div>

            <div class="mb-2">
              <label class="form-label">Quantity</label>
              <input class="form-control" name="quantity" type="number" min="1" step="1" required>
            </div>

            <div class="mb-3">
              <label class="form-label">Notes (optional)</label>
              <input class="form-control" name="notes" maxlength="255">
            </div>

            <button class="btn btn-danger w-100" type="submit">Save Stock OUT</button>
          </form>
        </div>
      </div>
    </div>

    <div class="col-12 col-lg-8">
      <div class="card shadow-sm">
        <div class="card-body">
          <h6 class="mb-3">Recent Stock OUT</h6>
          <div class="table-responsive">
            <table class="table table-sm align-middle">
              <thead>
                <tr>
                  <th>Date</th>
                  <th>SKU</th>
                  <th>Product</th>
                  <th class="text-end">Qty</th>
                </tr>
              </thead>
              <tbody>
                <?php if (!$recent): ?>
                  <tr><td colspan="4" class="text-muted">No records yet.</td></tr>
                <?php else: ?>
                  <?php foreach ($recent as $r): ?>
                    <tr>
                      <td><?= htmlspecialchars($r['created_at']) ?></td>
                      <td><?= htmlspecialchars($r['sku']) ?></td>
                      <td><?= htmlspecialchars($r['name']) ?></td>
                      <td class="text-end"><?= (int)$r['quantity'] ?></td>
                    </tr>
                  <?php endforeach; ?>
                <?php endif; ?>
              </tbody>
            </table>
          </div>

          
        </div>
      </div>
    </div>
  </div>
</main>

<?php require_once __DIR__ . '/../templates/footer.php'; ?>
